﻿
/****************************************************************************/
/*Copyright (c) 2011, Florent DEVILLE.                                      */
/*All rights reserved.                                                      */
/*                                                                          */
/*Redistribution and use in source and binary forms, with or without        */
/*modification, are permitted provided that the following conditions        */
/*are met:                                                                  */
/*                                                                          */
/* - Redistributions of source code must retain the above copyright         */
/*notice, this list of conditions and the following disclaimer.             */
/* - Redistributions in binary form must reproduce the above                */
/*copyright notice, this list of conditions and the following               */
/*disclaimer in the documentation and/or other materials provided           */
/*with the distribution.                                                    */
/* - The names of its contributors cannot be used to endorse or promote     */
/*products derived from this software without specific prior written        */
/*permission.                                                               */
/* - The source code cannot be used for commercial purposes without         */
/*its contributors' permission.                                             */
/*                                                                          */
/*THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       */
/*"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT         */
/*LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS         */
/*FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE            */
/*COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,       */
/*INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,      */
/*BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;          */
/*LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER          */
/*CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT        */
/*LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN         */
/*ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE           */
/*POSSIBILITY OF SUCH DAMAGE.                                               */
/****************************************************************************/

using GE.Visualisation;
using GE.Physics.Shapes;
using GE.Physics;
using GE.Manager;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;
using GE.TimeClock;

namespace GE.World.Entities
{
    class CutmanScissorEntity : WorldEntity
    {
        /// <summary>
        /// Animation id
        /// </summary>
        int _iIdAnimation;

        /// <summary>
        /// Animation's current frame
        /// </summary>
        int _iAnimationCurrentFrame;

        /// <summary>
        /// Animation's current time
        /// </summary>
        int _iAnimationCurrentTime;

        /// <summary>
        /// Target to hit;
        /// </summary>
        Vector2 _v2Target;

        /// <summary>
        /// Hit box
        /// </summary>
        DynamicShapeRectangle _shape;

        /// <summary>
        /// List of waypoint
        /// </summary>
        Vector2[] _v2Waypoint;

        /// <summary>
        /// Current waypoint used
        /// </summary>
        int _iCurrentWaypoint;

        /// <summary>
        /// Entity birth time
        /// </summary>
        int _iBirthTime;

        /// <summary>
        /// Entity lifetime
        /// </summary>
        const int LIFETIME = 500;

        /// <summary>
        /// Reference to the cutman entity
        /// </summary>
        CutmanEntity _refToCutmanEntity;

        public Vector2 Target { set { _v2Target = value; } }

        public CutmanEntity Cutman { set { _refToCutmanEntity = value; } }

        /// <summary>
        /// Constructor
        /// </summary>
        public CutmanScissorEntity()
            : base()
        {
            _iIdAnimation = -1;
            _iAnimationCurrentFrame = -1;
            _iAnimationCurrentTime = -1;
            _shape = Physics.Physics.Instance.createDynamicRectangle(0, 0, Vector2.Zero, this);
            _shape._iGroup = (int)ePhysicGroup.ePhysicEnemy;

            _v2Waypoint = new Vector2[3];
        }

        /// <summary>
        /// Initialise the entity
        /// </summary>
        public override void init()
        {
            _iIdAnimation = Visu.Instance.getAnimationID("Scissor");
            Animation animationScissor = Visu.Instance.getAnimation(_iIdAnimation);
            int idTexture = animationScissor.indexTexture;
            int idSprite = animationScissor.idFirstSprite;
            int iWidth = Visu.Instance.getSpriteWidth(idTexture, idSprite);
            int iHeight = Visu.Instance.getSpriteHeight(idTexture, idSprite);
            _shape.resize(iWidth, iHeight);
            base.init();
        }

        /// <summary>
        /// Activate the entity
        /// </summary>
        public override void activate()
        {
            base.activate();
            _iCurrentWaypoint = 0;
            _shape._bCollisionEnable = true;
            _iBirthTime = TimeClock.Clock.instance.millisecs;

            _v2Waypoint[0] = Position;
            _v2Waypoint[1] = _v2Target;
            _v2Waypoint[2] = _refToCutmanEntity.Position;
        }

        public override void update()
        {
            _v2Waypoint[2] = _refToCutmanEntity.Position;
            float fElapsedTime = TimeClock.Clock.instance.millisecs - _iBirthTime;
            float fRatio = fElapsedTime / LIFETIME;

            if (fRatio >= 1)
            {
                fRatio = 0;
                _iCurrentWaypoint++;
                _iBirthTime = TimeClock.Clock.instance.millisecs;
                if (_iCurrentWaypoint > 1)
                {
                    die();
                    _refToCutmanEntity.setHasScissor(true);
                    return;
                }
            }

            Vector2 newPosition = Vector2.Zero;
            switch (_iCurrentWaypoint)
            {
                case 0:
                    newPosition = Vector2.CatmullRom(_v2Waypoint[0], _v2Waypoint[0], _v2Waypoint[1], _v2Waypoint[2], fRatio);
                    break;

                case 1:
                    newPosition = Vector2.CatmullRom(_v2Waypoint[0], _v2Waypoint[1], _v2Waypoint[2], _v2Waypoint[2], fRatio);
                    break;
            }
            setPosition(newPosition);

            //check collision with player
            CollisionResult res = Physics.Physics.Instance.checkFirstRegisteredCollisionEx(_shape, (int)ePhysicGroup.ePhysicPlayer);
            if (res != null)
            {
                res.Entity.hurt(10);
            }
        }

        public override void render()
        {
            Visu.Instance.displayAnimation(_iIdAnimation, ScreenPosition, ref _iAnimationCurrentFrame, ref _iAnimationCurrentTime);
#if DEBUG
            Vector2[] obb = _shape.getOrientedBoundingBox();
            for (int i = 0; i < 4; i++)
                obb[i] -= World.Instance.CameraPosition;

            Visu.Instance.displayPolygon(obb);
#endif
        }

        public override void die()
        {
            
        }

        private void setPosition(Vector2 v)
        {
            Position = v;
            _shape._v2position = v;
        }
    }
}
